﻿using System.Collections;
using System.Collections.Generic;
using System.Threading.Tasks;
using UnityEngine;

public class HidableController : MonoBehaviour
{
    //Public editor variables
    public Sprite SpawnTileSprite;
    public Sprite DefaultSprite;
    public AnimationClip HideAnimation;
    public AnimationClip UnhideAnimation;
    public AnimationClip ConfirmHideAnimation;
    public AnimationClip RevealBeginAnimation;
    public AnimationClip RevealEndAnimation;
    public AudioClip RevealBeginAudio;
    public AudioClip RevealEndAudio;
    public bool IsOptionalHidable;
    
    private Animator _Animator;

    private TaskCompletionSource<bool> AnimCompleteTask;  //Task to allow waiting of page to close

    public void Start()
    {
        //Get the animator, override its controller with the animations set in the editor
        _Animator = gameObject.GetComponent<Animator>();
        AnimatorOverrideController controller = new AnimatorOverrideController(_Animator.runtimeAnimatorController);
        _Animator.runtimeAnimatorController = controller;
        controller["Hide"] = HideAnimation;
        controller["Unhide"] = UnhideAnimation;
        controller["Confirm"] = ConfirmHideAnimation;
        controller["RevealBegin"] = RevealBeginAnimation;
        controller["RevealEnd"] = RevealEndAnimation;
    }

    public void Awake()
    {
        AnimCompleteTask = new TaskCompletionSource<bool>();
    }

    /// <summary>
    /// Plays the hide animation
    /// </summary>
    /// <returns></returns>
    public async Task Hide()
    {
        _Animator.enabled = true;

        if (HideAnimation != null)
        {
            //Play the hide animation and wait for it to complete
            _Animator.Play("Hide", -1, 0.0f);
            await AnimCompleteTask.Task;
            AnimCompleteTask = new TaskCompletionSource<bool>();
        }

        else
        {
            //There's no hide animation, fallback to the defaults
            _Animator.ResetTrigger("IdleTrigger");
            _Animator.SetTrigger("GlowTrigger");
        }
    }

    /// <summary>
    /// Plays the unhide animation
    /// </summary>
    /// <returns></returns>
    public async Task Unhide()
    {
        _Animator.enabled = true;

        if (UnhideAnimation != null)
        {
            //Play the unhide animation and wait for it to complete
            _Animator.Play("Unhide", -1, 0.0f);
            await AnimCompleteTask.Task;
            AnimCompleteTask = new TaskCompletionSource<bool>();
        }
        
        //Animation done, reset
        _Animator.ResetTrigger("GlowTrigger");
        _Animator.SetTrigger("IdleTrigger");
        _Animator.Play("Idle", -1, 0.0f);
    }

    /// <summary>
    /// Plays the hide confirmation animation
    /// </summary>
    /// <returns></returns>
    public async Task Confirm()
    {
        _Animator.enabled = true;

        if (ConfirmHideAnimation != null)
        {
            //Play the confirm animation and wait for it to complete
            _Animator.Play("Confirm", -1, 0.0f);
            await AnimCompleteTask.Task;
            AnimCompleteTask = new TaskCompletionSource<bool>();
        }

        //Animation done, reset
        _Animator.ResetTrigger("GlowTrigger");
         _Animator.SetTrigger("IdleTrigger");
        _Animator.Play("Idle", -1, 0.0f);
        ResetHidable();
    }

    /// <summary>
    /// Plays the reveal begin and end animations as appropriate
    /// </summary>
    /// <param name="finalReveal">A boolean representing whether this with be the final winning reveal</param>
    /// <returns></returns>
    public async Task Reveal(bool finalReveal = false)
    {
        _Animator.enabled = true;

        if (AudioManager.Instance != null && RevealBeginAudio != null)
        {
            AudioManager.Instance.PlaySFX(RevealBeginAudio);
        }

        if (RevealBeginAnimation != null)
        {
            //Play the reveal animation and wait for it to complete
            _Animator.Play("RevealBegin", -1, 0.0f);
            await AnimCompleteTask.Task;
            AnimCompleteTask = new TaskCompletionSource<bool>();
        }

        else
        {
            if (!finalReveal)
            {
                //This isn't the final winning reveal and we've got no animation so fallback to the default flash
                _Animator.SetTrigger("FlashTrigger");
            }
        }

        if (!finalReveal)
        {
            //This isn't the final winning reveal so let's play the audio and animation if we can then reset to idle
            if (RevealEndAudio != null && AudioManager.Instance != null)
            {
                AudioManager.Instance.PlaySFX(RevealEndAudio);
            }

            if (RevealEndAnimation != null)
            {
                _Animator.Play("RevealEnd", -1, 0.0f);
                await AnimCompleteTask.Task;
                AnimCompleteTask = new TaskCompletionSource<bool>();
            }

            _Animator.Play("Idle", -1, 0.0f);
        }
    }

    /// <summary>
    /// Resets the hidable animations, sprites and triggers
    /// </summary>
    public void ResetHidable()
    {
        _Animator.enabled = true;
        gameObject.GetComponent<SpriteRenderer>().sprite = DefaultSprite;
       _Animator.ResetTrigger("GlowTrigger");
       _Animator.SetTrigger("IdleTrigger");
        _Animator.Play("Idle", -1, 0.0f);
    }

    /// <summary>
    /// Public editor function to be called on the last frame of an animation
    /// </summary>
    public void OnAnimationCompleted()
    {
        AnimCompleteTask.SetResult(true);
    }
}
